/**********************************************************************\
 © COPYRIGHT 2018 Corporation for National Research Initiatives (CNRI);
                        All rights reserved.

        The HANDLE.NET software is made available subject to the
      Handle.Net Public License Agreement, which may be obtained at
          http://hdl.handle.net/20.1000/112 or hdl:20.1000/112
\**********************************************************************/

package net.handle.server.servletcontainer.auth;

import java.security.PrivateKey;
import java.security.Signature;

import javax.servlet.http.HttpSession;

import org.apache.commons.codec.binary.Base64;

import net.handle.hdllib.AuthenticationInfo;
import net.handle.hdllib.ChallengeResponse;
import net.handle.hdllib.HandleException;
import net.handle.hdllib.Util;
import net.handle.server.servletcontainer.HandleServerInterface;

public class HandleAuthenticationStatus {
    private String sessionId;
    private byte[] nonce;
    private byte[] cnonce;
    private byte[] serverSignature;
    private String authorizationHeader;
    private AuthenticationInfo authInfo;
    private String id;

    public HandleAuthenticationStatus() {
    }

    public static byte[] generateNonce() {
        return ChallengeResponse.generateNonce();
    }

    public String getSessionId() {
        return sessionId;
    }

    public void setSessionId(String sessionId) {
        this.sessionId = sessionId;
    }

    public byte[] getNonce() {
        return nonce;
    }

    public void setNonce(byte[] nonce) {
        this.nonce = nonce;
    }

    public byte[] getCnonce() {
        return cnonce;
    }

    public void setCnonce(byte[] cnonce) {
        this.cnonce = cnonce;
    }

    public byte[] getServerSignature() {
        return serverSignature;
    }

    public void setServerSignature(byte[] serverSignature) {
        this.serverSignature = serverSignature;
    }

    public String getAuthorizationHeader() {
        return authorizationHeader;
    }

    public void setAuthorizationHeader(String authorizationHeader) {
        this.authorizationHeader = authorizationHeader;
    }

    public AuthenticationInfo getAuthInfo() {
        return authInfo;
    }

    public void setAuthInfo(AuthenticationInfo authInfo) {
        this.authInfo = authInfo;
    }

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public AuthenticationInfoWithId getAuthInfoWithId() {
        if (authInfo == null) return null;
        return new AuthenticationInfoWithId(id, authInfo);
    }

    public void setAuthInfoWithId(AuthenticationInfoWithId authInfo) {
        this.id = authInfo.getId();
        this.authInfo = authInfo.getAuthInfo();
    }

    public String getServerAlg(HandleServerInterface handleServer) throws HandleException {
        return Util.decodeString(Util.getHashAlgIdFromSigId(Util.getDefaultSigId(handleServer.getPrivateKey().getAlgorithm())));
    }

    public byte[] getServerSignature(HandleServerInterface handleServer, byte[] cnonceParam) throws HandleException {
        synchronized (this) {
            if (this.serverSignature != null && Util.equals(cnonceParam, this.cnonce)) {
                return this.serverSignature;
            }
        }
        PrivateKey privateKey = handleServer.getPrivateKey();
        try {
            Signature signer = Signature.getInstance(Util.getDefaultSigId(privateKey.getAlgorithm()));
            signer.initSign(privateKey);
            signer.update(this.nonce);
            signer.update(cnonceParam);
            byte[] signatureBytes = signer.sign();
            synchronized (this) {
                this.cnonce = cnonceParam;
                this.serverSignature = signatureBytes;
            }
            return signatureBytes;
        } catch (Exception e) {
            throw new HandleException(HandleException.INTERNAL_ERROR, "Unable to sign challenge", e);
        }
    }

    public static HandleAuthenticationStatus fromSession(HttpSession session, boolean create) {
        HandleAuthenticationStatus res = (HandleAuthenticationStatus) session.getAttribute(HandleAuthenticationStatus.class.getName());
        if (res != null) return res;
        if (!create) return null;
        Object sessionLock = getSessionLock(session);
        synchronized (sessionLock) {
            res = (HandleAuthenticationStatus) session.getAttribute(HandleAuthenticationStatus.class.getName());
            if (res != null) return res;
            res = new HandleAuthenticationStatus();
            res.setSessionId(session.getId());
            res.setNonce(generateNonce());
            session.setAttribute(HandleAuthenticationStatus.class.getName(), res);
            return res;
        }
    }

    private static final Object LOCK = new Object();

    private static Object getSessionLock(HttpSession session) {
        Object lock = session.getAttribute(HandleAuthenticationStatus.class.getName() + ".lock");
        if (lock != null) return lock;
        synchronized (LOCK) {
            lock = session.getAttribute(HandleAuthenticationStatus.class.getName() + ".lock");
            if (lock != null) return lock;
            lock = new Object();
            session.setAttribute(HandleAuthenticationStatus.class.getName() + ".lock", lock);
            return lock;
        }
    }

    public static void processServerSignature(HandleServerInterface handleServer, HttpSession session, HandleAuthorizationHeader handleAuthHeader, AuthenticationResponse authResp) {
        if (handleAuthHeader != null && handleAuthHeader.isRequestingServerSignature() && handleServer != null) {
            HandleAuthenticationStatus status = HandleAuthenticationStatus.fromSession(session, true);
            authResp.setSessionId(status.getSessionId());
            authResp.setNonce(status.getNonce());
            try {
                authResp.setServerSignature(status.getServerSignature(handleServer, Base64.decodeBase64(handleAuthHeader.getCnonce())));
                authResp.setServerAlg(status.getServerAlg(handleServer));
            } catch (HandleException e) {
                authResp.getErrors().add(e.toString());
            }
        }
    }

}
