/**********************************************************************\
 © COPYRIGHT 2018 Corporation for National Research Initiatives (CNRI);
                        All rights reserved.

        The HANDLE.NET software is made available subject to the
      Handle.Net Public License Agreement, which may be obtained at
          http://hdl.handle.net/20.1000/112 or hdl:20.1000/112
\**********************************************************************/

package net.handle.hdllib.trust;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import net.handle.hdllib.Common;
import net.handle.hdllib.HandleRecord;
import net.handle.hdllib.HandleResolver;
import net.handle.hdllib.HandleValue;
import net.handle.hdllib.Util;

public class HandleRecordTrustVerifier {

    private final ChainBuilder chainBuilder;
    private final ChainVerifier chainVerifier;
    private boolean isThrowing;

    public HandleRecordTrustVerifier(HandleResolver resolver) {
        this.chainBuilder = new ChainBuilder(resolver);
        this.chainVerifier = new ChainVerifier(resolver.getConfiguration().getRootKeys());
    }

    public HandleRecordTrustVerifier(ChainBuilder chainBuilder, ChainVerifier chainVerifier) {
        this.chainBuilder = chainBuilder;
        this.chainVerifier = chainVerifier;
    }

    public void setThrowing(boolean isThrowing) {
        this.isThrowing = isThrowing;
    }

    public boolean validateHandleRecord(HandleRecord handleRecord) throws TrustException {
        List<HandleValue> valuesList = handleRecord.getValues();
        Set<Integer> donaVerifiedValues = new HashSet<>();
        Set<Integer> localCertVerifiedValues = new HashSet<>();
        boolean localCertNeeded = false;
        try {
            List<JsonWebSignature> signatures = getJsonWebSignaturesFromValues(handleRecord.getValuesAsArray());
            if (signatures.isEmpty()) {
                if (isThrowing) throw new TrustException("No signatures");
                return false;
            }
            for (JsonWebSignature jws : signatures) {
                List<IssuedSignature> chain = chainBuilder.buildChain(jws);
                ChainVerificationReport report = chainVerifier.verifyValues(handleRecord.getHandle(), valuesList, chain);
                if (!report.canTrust()) { //Chain signatures verify all the way to DONA
                    if (isThrowing) throw new TrustException("Chain not trusted");
                    return false;
                }
                if (!report.valuesReport.badDigestValues.isEmpty()) {
                    if (isThrowing) throw new TrustException("Bad digests");
                    return false;
                }
                if (!report.valuesReport.missingValues.isEmpty()) {
                    if (isThrowing) throw new TrustException("Missing values");
                    return false;
                }
                localCertNeeded = report.chainNeedsRequiredSigner;
                if (report.canTrustAndAuthorized()) { //Are there no permissions problems in the DONA sense
                    donaVerifiedValues.addAll(report.valuesReport.verifiedValues);
                }
                if (report.canTrustAndAuthorizedUpToRequiredSigner()) { //Are there no permissions problems in the local cert sense
                    localCertVerifiedValues.addAll(report.valuesReport.verifiedValues);
                }
            }
        } catch (TrustException e) {
            if (isThrowing) throw e;
            return false;
        }
        if (isAllValuesThatNeedSignatureInVerifiedSet(handleRecord, donaVerifiedValues)) {
            if (!localCertNeeded) {
                return true;
            }
            // local cert is needed so better all verified values are in there
            if (isAllValuesThatNeedSignatureInVerifiedSet(handleRecord, localCertVerifiedValues)) {
                return true;
            } else {
                if (isThrowing) throw new TrustException("Failure of required signer");
                return false;
            }
        } else {
            if (isThrowing) throw new TrustException("Unsigned values");
            return false;
        }
    }

    private List<JsonWebSignature> getJsonWebSignaturesFromValues(HandleValue[] newValues) throws TrustException {
        HandleValue[] hsSignatureValues = getSignatureValues(newValues);
        List<JsonWebSignature> signatures = new ArrayList<>();
        if (hsSignatureValues == null) return signatures;
        for (HandleValue hsSignatureValue : hsSignatureValues) {
            JsonWebSignature signature = JsonWebSignatureFactory.getInstance().deserialize(hsSignatureValue.getDataAsString());
            signatures.add(signature);
        }
        return signatures;
    }

    private static boolean isAllValuesThatNeedSignatureInVerifiedSet(HandleRecord newRecord, Set<Integer> verifiedValues) {
        for (HandleValue value : newRecord.getValues()) {
            int index = value.getIndex();
            if (!verifiedValues.contains(index)) {
                if (valueNeedsSignature(newRecord.getHandle(), value)) {
                    return false;
                }
            }
        }
        return true;
    }

    @SuppressWarnings("deprecation")
    private static boolean valueNeedsSignature(String handle, HandleValue value) {
        if (value.hasType(Common.HS_SIGNATURE_TYPE)) return false;
        if (!value.getAnyoneCanRead()) return false;
        if ("0.NA/0.NA".equalsIgnoreCase(handle) && (value.hasType(net.handle.hdllib.HandleSignature.SIGNATURE_TYPE) || value.hasType(net.handle.hdllib.HandleSignature.METADATA_TYPE))) return false;
        return true;
    }

    private static HandleValue[] getSignatureValues(HandleValue[] newValues) {
        return Util.filterValues(newValues, null, Common.HS_SIGNATURE_TYPE_LIST);
    }
}
